//
//  DataCenter.swift
//  Do It
//
//  Created by Jim Dovey on 10/13/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import Foundation
import Combine

fileprivate let jsonDataName = "todoData.json"

class DataCenter: ObservableObject {
    @Published var todoItems: [TodoItemStruct] = []
    @Published var todoLists: [TodoItemListStruct] = []
    private var saveCancellable: AnyCancellable?

    /// Queue used to perform modifications that would affect the set of included item identifiers (add, remove).
    private var mutationQ = DispatchQueue(label: "DataCenter.todoItemsMutationQueue")

    init() {
        let todoData: TodoData
        do {
            todoData = try loadJSON(from: jsonDataURL)
        } catch {
            todoData = createDefaultItems()
        }
        self.todoItems = todoData.items
        self.todoLists = todoData.lists
        saveWhenChanged()
    }

    private func createDefaultItems() -> TodoData {
        let encoder = JSONEncoder()
        encoder.dateEncodingStrategy = .iso8601
        do {
            let data = try encoder.encode(defaultTodoData)
            writeData(data)
        } catch {
            fatalError("Failed to create default todo items! \(error)")
        }
        return defaultTodoData
    }

    private lazy var jsonDataURL: URL = {
        let baseURL: URL
        do {
            baseURL = try FileManager.default.url(
                for: .documentDirectory, in: .userDomainMask,
                appropriateFor: nil, create: true)
        } catch {
            let homeURL = URL(fileURLWithPath: NSHomeDirectory(),
                              isDirectory: true)
            baseURL = homeURL.appendingPathComponent("Documents")
        }
        return baseURL.appendingPathComponent(jsonDataName)
    }()
}

// MARK: List/Item Lookups

extension DataCenter {
    @inlinable
    func list(for item: TodoItemStruct) -> TodoItemListStruct {
        guard let list = todoLists.first(where: { $0.id == item.listID }) else {
            preconditionFailure("No matching list for ID: \(item.listID)")
        }
        return list
    }

    @inlinable
    func items(in list: TodoItemListStruct) -> [TodoItemStruct] {
        todoItems.filter { $0.listID == list.id }
    }

    @inlinable
    func items<S: Sequence>(withIDs ids: S) -> [TodoItemStruct] where S.Element == Int {
        ids.compactMap { id in
            todoItems.first { $0.id == id }
        }
    }
}

// MARK: TodoItem Array Convenience

extension DataCenter {
    func addTodoItem(_ item: TodoItemStruct) {
        mutationQ.sync {
            let highestId = todoItems.map({ $0.id }).max() ?? 0
            var newItem = item
            newItem.id = highestId + 1
            todoItems.append(newItem)
        }
    }

    func removeTodoItems(atOffsets offsets: IndexSet) {
        mutationQ.sync {
            todoItems.remove(atOffsets: offsets)
        }
    }

    func updateTodoItem(_ item: TodoItemStruct) {
        if let idx = todoItems.firstIndex(where: { $0.id == item.id }) {
            todoItems[idx] = item
        }
        else if item.id == Int.min {
            addTodoItem(item)
        }
    }

    func moveTodoItems(fromOffsets offsets: IndexSet, to index: Int) {
        todoItems.move(fromOffsets: offsets, toOffset: index)
    }

    func addList(_ list: TodoItemListStruct) {
        mutationQ.sync {
            let highestId = todoLists.map({ $0.id }).max() ?? 0
            var newList = list
            newList.id = highestId + 1
            todoLists.append(newList)
        }
    }

    func removeLists(atOffsets offsets: IndexSet) {
        mutationQ.sync {
            todoLists.remove(atOffsets: offsets)
        }
    }

    func updateList(_ list: TodoItemListStruct) {
        if let idx = todoLists.firstIndex(where: { $0.id == list.id }) {
            todoLists[idx] = list
        }
        else if list.id == Int.min {
            addList(list)
        }
    }

    func moveLists(fromOffsets offsets: IndexSet, to index: Int) {
        todoLists.move(fromOffsets: offsets, toOffset: index)
    }
}

// MARK: Save/Load Functionality

extension DataCenter {
    private func saveWhenChanged() {
        let encoder = JSONEncoder()
        encoder.dateEncodingStrategy = .iso8601

        let scheduler = UIBackgroundTaskScheduler(
            "Saving To-Do Items", target: DispatchQueue.global())

        self.saveCancellable = $todoItems
            // Combine list and items into a single TodoData object
            .combineLatest($todoLists) { TodoData(lists: $1, items: $0) }
            // Ignore the first value from this publisher; that's the initial
            // content that we started with, and we don't want to write that
            // out.
            .dropFirst()
            // Require there to be no changes for 100 ms before proceeding
            .debounce(for: .milliseconds(100), scheduler: scheduler)
            // Encode the TodoData using our JSONEncoder
            .encode(encoder: encoder)
            // When the encoder's output arrives, pipe it into the
            // writeData function to send it to disk.
            .sink(receiveCompletion: { _ in }, receiveValue: self.writeData)
    }

    private func writeData(_ data: Data) {
        do {
            try data.write(to: self.jsonDataURL,
                           options: [.completeFileProtection, .atomic])
        } catch {
            NSLog("Error writing JSON data: \(error)")
        }
    }

    private func loadJSON<T: Decodable>(
        from url: URL,
        as type: T.Type = T.self
    ) throws -> T {
        let data = try Data(contentsOf: url,
                            options: [.mappedIfSafe])

        let decoder = JSONDecoder()
        decoder.dateDecodingStrategy = .iso8601
        return try decoder.decode(type, from: data)
    }
}
